<?php

/**
 * @property string testmode
 * @property string merchantid
 * @property string paymentgatewayid
 * @property string secretkey
 * @property string langpaymentpage
 * @property string successurl
 * @property string cancelurl
 * @property string errorurl
 * @property string notification_email
 */
class WC_Gateway_Borgun extends WC_Payment_Gateway {

	const BORGUN_ENDPOINT_SANDBOX = 'https://test.borgun.is/securepay/';
	const BORGUN_ENDPOINT_LIVE = 'https://securepay.borgun.is/securepay/';

	/**
	 * Logger instance
	 *
	 * @var WC_Logger
	 */
	public static $log = false;

	/**
	 * Debug
	 *
	 * @var bool
	 */
	private $debug;

	/**
	 * Whether or not logging is enabled
	 *
	 * @var bool
	 */
	public static $log_enabled = false;

	/**
	 * Gateway testmode
	 *
	 * @var string
	 */
	private $testmode;

	/**
	 * MerchantId that identifies the merchant
	 *
	 * @var string
	 */
	private $merchantid;

	/**
	 * Payment Gateway Id that identifies the payment method
	 *
	 * @var string
	 */
	private $paymentgatewayid;

	/**
	 * Saltpay secretkey
	 *
	 * @var string
	 */
	private $secretkey;

	/**
	 * Language of Payment Page
	 *
	 * @var string
	 */
	private $langpaymentpage;

	/**
	 * Receipt text
	 *
	 * @var string
	 */
	private $receipttext;

	/**
	 * Redirect text
	 *
	 * @var string
	 */
	private $redirecttext;

	/**
	 * Skip receipt page
	 *
	 * @var string
	 */
	private $skipreceiptpage;

	/**
	 * Success Page URL
	 *
	 * @var string
	 */
	private $successurl;

	/**
	 * Cancel Page URL
	 *
	 * @var string
	 */
	private $cancelurl;

	/**
	 * Error Page URL
	 *
	 * @var string
	 */
	private $errorurl;

	/**
	 * Notification Email
	 *
	 * @var string
	 */
	private $notification_email;

	/**
	 * Order line items grouping
	 *
	 * @var string
	 */
	private $TotalLineItem;

	public function __construct() {
		$this->id                 = 'borgun';
		$this->icon               = BORGUN_URL . '/teya.png';
		$this->has_fields         = false;
		$this->method_title       =  __('Teya', 'borgun_woocommerce');
		$this->method_description = __('Teya Secure Payment Page enables merchants to sell products securely on the web with minimal integration effort', 'borgun_woocommerce');

		// What methods do support plugin
		$this->supports = array(
			'products',
			'refunds',
		);


		// Load the form fields
		$this->init_form_fields();
		$this->init_settings();
		// Get setting values
		$this->enabled            = $this->get_option( 'enabled' );
		$this->title              = $this->get_option( 'title' );
		$this->description        = $this->get_option( 'description' );
		$this->testmode           = $this->get_option( 'testmode' );
		$this->merchantid         = $this->get_option( 'merchantid' );
		$this->paymentgatewayid   = $this->get_option( 'paymentgatewayid' );
		$this->secretkey          = $this->get_option( 'secretkey' );
		$this->langpaymentpage    = $this->get_option( 'langpaymentpage' );
		$this->receipttext 		  = $this->get_option( 'receipttext', __('Thank you - your order is now pending payment. We are now redirecting you to Teya to make payment.', 'borgun_woocommerce') );
		$this->redirecttext 	  = $this->get_option( 'redirecttext',  __('Thank you for your order. We are now redirecting you to Teya to make payment.', 'borgun_woocommerce') );
		$this->skipreceiptpage    = 'yes' === $this->get_option( 'skipreceiptpage', 'no' );
		$this->successurl         = $this->get_option( 'successurl' );
		$this->cancelurl          = $this->get_option( 'cancelurl' );
		$this->errorurl           = $this->get_option( 'errorurl' );
		$this->notification_email = $this->get_option( 'notification_email' );
		$this->TotalLineItem      = 'yes' === $this->get_option( 'TotalLineItem', 'no' );
		$this->debug              = 'yes' === $this->get_option( 'debug', 'no' );
		self::$log_enabled        = $this->debug;
		// Filters
		add_filter( 'wcml_gateway_text_keys_to_translate', array( $this, 'borgun_text_keys_to_translate' ) );
		// Hooks
		add_action( 'woocommerce_receipt_' . $this->id, array( $this, 'receipt_page' ) );
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_api_wc_gateway_borgun', array( $this, 'handle_webhook_callback' ) );
		add_action( 'woocommerce_thankyou_' . $this->id, array( $this, 'success_page_redirect' ) );
		add_action( 'before_woocommerce_pay', array( $this, 'checkout_payment_handler'), 9 );
		if ( ! $this->is_valid_for_use() ) {
			$this->enabled = false;
		}
	}

	/**
	 * Logging method.
	 *
	 * @param string $message Log message.
	 * @param string $level Optional. Default 'info'. Possible values:
	 *                      emergency|alert|critical|error|warning|notice|info|debug.
	 */
	public static function log( $message, $level = 'info' ) {
		if (self::$log_enabled) {
			if ( empty( self::$log ) ) {
				self::$log = wc_get_logger();
			}
			self::$log->log( $level, $message, array( 'source' => 'borgun' ) );
		}
	}

	public function admin_options() {
		?>
		<h3> <?php esc_html_e( 'Teya', 'borgun_woocommerce' ); ?></h3>
		<p> <?php esc_html_e( 'Pay with your credit card via Teya.', 'borgun_woocommerce' ); ?></p>
		<?php if ( $this->is_valid_for_use() ) : ?>
			<table class="form-table"><?php $this->generate_settings_html(); ?></table>
		<?php else : ?>
			<div class="inline error"><p><strong><?php esc_html_e( 'Gateway Disabled:', 'borgun_woocommerce' ); ?></strong> <?php esc_html_e( 'Current Store currency is not supported by Teya SecurePay. Allowed values are GBP, USD, EUR, DKK, NOK, SEK, CHF, CAD, HUF, BHD, AUD, RUB, PLN, RON, HRK, CZK and ISK.', 'borgun_woocommerce' ); ?></p></div>
			<?php
		endif;
	}

	//Check if this gateway is enabled and available in the user's country
	function is_valid_for_use() {
		if ( ! in_array( get_woocommerce_currency(), array(
			'ISK',
			'GBP',
			'USD',
			'EUR',
			'DKK',
			'NOK',
			'SEK',
			'CHF',
			'CAD',
			'HUF',
			'BHD',
			'AUD',
			'RUB',
			'PLN',
			'RON',
			'HRK',
			'CZK',
		) )
		) {
			return false;
		}

		return true;
	}

	//Initialize Gateway Settings Form Fields
	function init_form_fields() {
		$this->form_fields = array(
			'enabled'            => array(
				'title'       => __( 'Enable/Disable', 'borgun_woocommerce' ),
				'label'       => __( 'Enable Teya', 'borgun_woocommerce' ),
				'type'        => 'checkbox',
				'description' => '',
				'default'     => 'no'
			),
			'title'              => array(
				'title'       =>  __( 'Title', 'borgun_woocommerce' ),
				'type'        => 'text',
				'description' =>  __( 'This controls the title which the user sees during checkout.', 'borgun_woocommerce' ),
				'default'     =>  __( 'Teya', 'borgun_woocommerce' )
			),
			'description'        => array(
				'title'       => __( 'Description', 'borgun_woocommerce' ),
				'type'        => 'textarea',
				'description' => __( 'This controls the description which the user sees during checkout.', 'borgun_woocommerce' ),
				'default'     => __( 'Pay with your credit card via Teya.', 'borgun_woocommerce' )
			),
			'testmode'           => array(
				'title'       => __( 'Teya Test Mode', 'borgun_woocommerce' ),
				'label'       => __( 'Enable Test Mode', 'borgun_woocommerce' ),
				'type'        => 'checkbox',
				'description' => __( 'Place the payment gateway in development mode.', 'borgun_woocommerce' ),
				'default'     => 'no'
			),
			'paymentgatewayid'   => array(
				'title'       => __( 'Payment Gateway ID', 'borgun_woocommerce' ),
				'type'        => 'text',
				'description' => __( 'This is the Payment Gateway ID supplied by Teya.', 'borgun_woocommerce' ),
				'default'     => '16'
			),
			'merchantid'         => array(
				'title'       => __( 'Merchant ID', 'borgun_woocommerce' ),
				'type'        => 'text',
				'description' =>  __( 'This is the ID supplied by Teya.', 'borgun_woocommerce' ),
				'default'     => '9275444'
			),
			'secretkey'          => array(
				'title'       => __( 'Secret Key', 'borgun_woocommerce' ),
				'type'        => 'text',
				'description' => __( 'This is the Secret Key supplied by Teya.', 'borgun_woocommerce' ),
				'default'     => '99887766'
			),
			'notification_email' => array(
				'title'       => __( 'Notification Email', 'borgun_woocommerce' ),
				'type'        => 'text',
				'description' => __( 'This is the email Teya will send payment receipts to.', 'borgun_woocommerce' ),
				'default'     => get_option( 'admin_email' )
			),
			'langpaymentpage'    => array(
				'title'       => __( 'Language of Payment Page', 'borgun_woocommerce' ),
				'type'        => 'select',
				'description' => __('Select which language to show on Payment Page.', 'borgun_woocommerce' ),
				'default'     => 'en',
				'options'     => array(
					'is' => __('Icelandic', 'borgun_woocommerce' ),
					'en' => __('English', 'borgun_woocommerce' ),
					'de' => __('German', 'borgun_woocommerce' ),
					'fr' => __('French', 'borgun_woocommerce' ),
					'ru' => __('Russian', 'borgun_woocommerce' ),
					'es' => __('Spanish', 'borgun_woocommerce' ),
					'it' => __('Italian', 'borgun_woocommerce' ),
					'pt' => __('Portugese', 'borgun_woocommerce' ),
					'si' => __('Slovenian', 'borgun_woocommerce' ),
					'hu' => __('Hungarian', 'borgun_woocommerce' ),
					'se' => __('Swedish', 'borgun_woocommerce' ),
					'nl' => __('Dutch', 'borgun_woocommerce' ),
					'pl' => __('Polish', 'borgun_woocommerce' ),
					'no' => __('Norwegian', 'borgun_woocommerce' ),
					'cz' => __('Czech', 'borgun_woocommerce' ),
					'sk' => __('Slovak', 'borgun_woocommerce' ),
					'hr' => __('Hrvatski', 'borgun_woocommerce' ),
					'ro' => __('Romanian', 'borgun_woocommerce' ),
					'dk' => __('Danish', 'borgun_woocommerce' ),
					'fi' => __('Finnish', 'borgun_woocommerce' ),
					'fo' => __('Faroese', 'borgun_woocommerce' ),
					'sr' => __('Serbian', 'borgun_woocommerce' ),
					'bg' => __('Bulgarian', 'borgun_woocommerce' ),
					'lt' => __('Lithuanian', 'borgun_woocommerce' ),
				)
			),
			'receipttext'     	=> array(
				'title'       =>  __('Receipt text', 'borgun_woocommerce'),
				'type'        => 'textarea',
				'description' =>  __('Buyer will see this text after woocommerce order create.', 'borgun_woocommerce'),
				'default'     =>  __('Thank you - your order is now pending payment. We are now redirecting you to Teya to make payment.', 'borgun_woocommerce'),
			),
			'redirecttext'     	=> array(
				'title'       =>  __('Redirect text', 'borgun_woocommerce'),
				'type'        => 'textarea',
				'description' => __('Buyer will see this text before redirecting to Teya', 'borgun_woocommerce'),
				'default'     => __('Thank you for your order. We are now redirecting you to Teya to make payment.', 'borgun_woocommerce'),
			),
			'skipreceiptpage'=> array(
				'title'       =>  __('Skip Teya receipt page', 'borgun_woocommerce'),
				'type'        => 'checkbox',
				'description' => __('If checked Teya receipt page is not displayed and the buyer is redirected to the Success Page URL upon successful payment.', 'borgun_woocommerce'),
				'default'     => 'yes'
			),
			'successurl'         => array(
				'title'       =>  __('Success Page URL', 'borgun_woocommerce' ),
				'type'        => 'text',
				'description' =>  __('Buyer will be sent to this page after a successful payment.', 'borgun_woocommerce' ),
				'default'     => ''
			),
			'cancelurl'          => array(
				'title'       =>  __('Cancel Page URL', 'borgun_woocommerce' ),
				'type'        => 'text',
				'description' =>  __('Buyer will be sent to this page if he pushes the cancel button instead of finalizing the payment.', 'borgun_woocommerce' ),
				'default'     => ''
			),
			'errorurl'           => array(
				'title'       =>  __('Error Page URL', 'borgun_woocommerce' ),
				'type'        => 'text',
				'description' =>  __('Buyer will be sent to this page if an unexpected error occurs.', 'borgun_woocommerce' ),
				'default'     => ''
			),
			'TotalLineItem' => array(
				'title'       => __('Order line items grouping', 'borgun_woocommerce'),
				'label'       => __('Send order as 1 line item', 'borgun_woocommerce'),
				'type'        => 'checkbox',
				'description' => __('You can uncheck this if you don\'t use discounts and have integer quantities.', 'borgun_woocommerce'),
				'default'     => 'yes'
			),
			'debug' => array(
				'title'       => __( 'Debug', 'borgun_woocommerce' ),
				'label'       => __( 'Enable Debug Mode', 'borgun_woocommerce' ),
				'type'        => 'checkbox',
				'default'     => 'no',
				'desc_tip'    => true
			)
		);
	}

	/**
	 * Get gateway icon.
	 *
	 * @return string
	 */
	public function get_icon() {
		$icon_html = '<img class="wc-borgun-payment-gateway-checkout-logo" src="' . $this->icon . '" alt="' . esc_html( $this->get_method_title() ) . '" />';
		return apply_filters( 'borgun_gateway_icon', $icon_html, $this->id );
	}

	/**
	 * @param WC_Order $order
	 *
	 * @return false|string
	 */
	function check_hash( $order, $payment_return_url='') {
		$currency = $order->get_currency();
		$order_total = number_format( $order->get_total(), $this->get_price_decimals($currency), '.', '' );
		$ipnUrl           = WC()->api_request_url( 'WC_Gateway_Borgun' );
		$hash             = array();
		$hash[]           = $this->merchantid;
		$hash[]           = $payment_return_url;
		$hash[]           = $ipnUrl;
		$hash[]           = 'WC-' . $order->get_id();
		$hash[]           = $order_total;
		$hash[]           = $currency;
		$hash = apply_filters( 'borgun_'.$this->id.'_check_hash', $hash, $order );
		$message          = implode( '|', $hash );

		$CheckHashMessage = trim( $message );
		if (extension_loaded('mbstring')) {
			$CheckHashMessage = mb_convert_encoding($CheckHashMessage, 'UTF-8');
		}else{
			$CheckHashMessage = utf8_encode($CheckHashMessage);
		}
		$Checkhash = hash_hmac( 'sha256', $CheckHashMessage, $this->secretkey );



		return $Checkhash;
	}

	/**
	 * @param WC_Order $order
	 *
	 * @return false|string
	 */
	function check_order_hash( $order ) {
		$currency = $order->get_currency();
		$order_total = number_format( $order->get_total(), $this->get_price_decimals($currency), '.', '' );
		$hash             = array();
		$hash[]           = 'WC-' . $order->get_id();
		$hash[]           = $order_total;
		$hash[]           = $currency;
		$hash = apply_filters( 'borgun_'.$this->id.'_check_order_hash', $hash, $order );
		$message          = implode( '|', $hash );
		$CheckHashMessage = trim( $message );

		if (extension_loaded('mbstring')) {
			$CheckHashMessage = mb_convert_encoding($CheckHashMessage, 'UTF-8');
		}else{
			$CheckHashMessage = utf8_encode($CheckHashMessage);
		}
		$Checkhash = hash_hmac( 'sha256', $CheckHashMessage, $this->secretkey );

		return $Checkhash;
	}

	/**
	 * @param WC_Order $order
	 *
	 * @return false|string
	 */
	function check_order_refund_hash( $order ) {
		$hash             = array();
		$hash[]           = $this->merchantid;
		$refundid = $order->get_meta('_' . $this->id . '_refundid');
		$hash[]           = $refundid;
		$hash = apply_filters( 'borgun_'.$this->id.'_check_order_refund_hash', $hash, $order );
		$message          = implode( '|', $hash );
		$CheckHashMessage = trim( $message );
		if (extension_loaded('mbstring')) {
			$CheckHashMessage = mb_convert_encoding($CheckHashMessage, 'UTF-8');
		}else{
			$CheckHashMessage = utf8_encode($CheckHashMessage);
		}
		$Checkhash = hash_hmac( 'sha256', $CheckHashMessage, $this->secretkey );

		return $Checkhash;
	}

	/**
	 * @param WC_Order $order
	 *
	 * @return array
	 */
	function get_borgun_args( $order ) {
		//Teya Args
		global $wp_version;
		$ipnUrl = WC()->api_request_url( 'WC_Gateway_Borgun' );
		$currency = $order->get_currency();
		$decimals = $this->get_price_decimals($currency);
		$order_total = number_format( $order->get_total(), $decimals, '.', '' );

		$payment_return_url = esc_url_raw($this->get_return_url( $order ));

		$borgun_args = array(
			'merchantid'             => $this->merchantid,
			'paymentgatewayid'       => $this->paymentgatewayid,
			'checkhash'              => $this->check_hash( $order, $payment_return_url ),
			'orderid'                => 'WC-' . $order->get_id(),
			'reference'              => $order->get_order_number(),
			'currency'               => $currency,
			'language'               => $this->langpaymentpage,
			'SourceSystem'           => 'WP' . $wp_version . ' - WC' . WC()->version . ' - BRG' . BORGUN_VERSION,
			'buyeremail'             => $order->get_billing_email(),
			'returnurlsuccess'       => $payment_return_url,
			'returnurlsuccessserver' => $ipnUrl,
			'returnurlcancel'        =>$order->get_checkout_payment_url( true ),
			'returnurlerror'         => $order->get_checkout_payment_url( true ),
			'amount'                 => $order_total,
			'pagetype'               => '0',
			'skipreceiptpage'        => ($this->skipreceiptpage) ? '1' : '0',
			'merchantemail'          => $this->notification_email,
		);
		$borgun_args = apply_filters( 'borgun_get_'.$this->id.'_args', $borgun_args, $order );
		// Cart Contents
		$total_line_item = $this->TotalLineItem;
		$include_tax = $this->tax_display();
		$item_loop = 0;
		if ( sizeof( $order->get_items( array( 'line_item', 'fee' ) ) ) > 0 ) {
			if ( $total_line_item == "yes" ) {
				$item_description = '';
				foreach ( $order->get_items( array( 'line_item', 'fee' ) ) as $item ) {
					$item_name = wp_strip_all_tags( $item->get_name() );
					if( !empty($item_description) ) $item_description .= ', ';
					$item_description .= $item_name;
				}
				if (strlen($item_description) > 499) $item_description = substr($item_description, 0, 496) . '...';
				$borgun_args[ 'itemdescription_' . $item_loop ] = html_entity_decode( $item_description, ENT_NOQUOTES, 'UTF-8' );
				$borgun_args[ 'itemcount_' . $item_loop ]       = 1;
				$borgun_args[ 'itemunitamount_' . $item_loop ]  = $borgun_args['amount'];
				$borgun_args[ 'itemamount_' . $item_loop ]      = $borgun_args['amount'];
			}else{
				foreach ( $order->get_items( array( 'line_item', 'fee' ) ) as $item ) {
					if ( 'fee' === $item['type'] ) {
						$fee = $item->get_total();
						if ( $include_tax && $this->fee_tax_display($item) ){
							$fee += $item->get_total_tax();
						}
						$fee_total = $this->round( $fee, $order );
						$item_name = wp_strip_all_tags( $item->get_name() );
						$borgun_args[ 'itemdescription_' . $item_loop ] = html_entity_decode( $item_name, ENT_NOQUOTES, 'UTF-8' );
						$borgun_args[ 'itemcount_' . $item_loop ]       = 1;
						$borgun_args[ 'itemunitamount_' . $item_loop ]  = $fee_total;
						$borgun_args[ 'itemamount_' . $item_loop ]      = $fee_total;

						$item_loop ++;
					}
					if ( $item['qty'] ) {
						$item_name = $item['name'];
						if ( $meta = wc_display_item_meta( $item ) ) {
							$item_name .= ' ( ' . $meta . ' )';
						}
						$item_name = wp_strip_all_tags($item_name);
						$item_subtotal = number_format( $order->get_item_subtotal( $item, $include_tax ), $decimals, '.', '' );
						$itemamount = $item_subtotal * $item['qty'];
						$borgun_args[ 'itemdescription_' . $item_loop ] = html_entity_decode( $item_name, ENT_NOQUOTES, 'UTF-8' );
						$borgun_args[ 'itemcount_' . $item_loop ]       = $item['qty'];
						$borgun_args[ 'itemunitamount_' . $item_loop ]  = number_format( $item_subtotal, $decimals, '.', '' );
						$borgun_args[ 'itemamount_' . $item_loop ]      = number_format( $itemamount, $decimals, '.', '' );
						$item_loop ++;
					}
				}
				if ( $order->get_shipping_total() > 0 ) {
					$shipping_total = $order->get_shipping_total();
					if( $include_tax ) $shipping_total += $order->get_shipping_tax();
					$shipping_total = $this->round( $shipping_total, $order );
					$borgun_args[ 'itemdescription_' . $item_loop ] = sprintf( /* translators: %s: Shipping */ __('Shipping (%s)', 'borgun_woocommerce' ), $order->get_shipping_method() );
					$borgun_args[ 'itemcount_' . $item_loop ]       = 1;
					$borgun_args[ 'itemunitamount_' . $item_loop ]  = number_format( $shipping_total, $decimals, '.', '' );
					$borgun_args[ 'itemamount_' . $item_loop ]      = number_format( $shipping_total, $decimals, '.', '' );
					$item_loop ++;
				}
				if (!$include_tax && $order->get_total_tax() > 0){
					$borgun_args[ 'itemdescription_' . $item_loop ] = __('Taxes', 'borgun_woocommerce' );
					$borgun_args[ 'itemcount_' . $item_loop ]       = 1;
					$borgun_args[ 'itemunitamount_' . $item_loop ]  = number_format( $order->get_total_tax(), $decimals, '.', '' );
					$borgun_args[ 'itemamount_' . $item_loop ]      = number_format( $order->get_total_tax(), $decimals, '.', '' );
					$item_loop ++;
				}
				if ( $order->get_total_discount() > 0 ) {
					$total_discount = $order->get_total_discount();
/*				Woocommerce can see any tax adjustments made thus far using subtotals.
					Since Woocommerce 3.2.3*/
					if(wc_tax_enabled() && method_exists('WC_Discounts','set_items') && $include_tax){
						$total_discount += $order->get_discount_tax();
					}
					if(wc_tax_enabled() && !method_exists('WC_Discounts','set_items') && !$include_tax){
						$total_discount -= $order->get_discount_tax();
					}

					$total_discount = $this->round($total_discount, $order);
					$borgun_args[ 'itemdescription_' . $item_loop ] = __('Discount', 'borgun_woocommerce' );
					$borgun_args[ 'itemcount_' . $item_loop ]       = 1;
					$borgun_args[ 'itemunitamount_' . $item_loop ]  = - number_format( $total_discount, $decimals, '.', '' );
					$borgun_args[ 'itemamount_' . $item_loop ]      = - number_format( $total_discount, $decimals, '.', '' );
					$item_loop ++;
				}
			}
		}

		return $borgun_args;
	}

	//Generate the borgun button link
	function generate_borgun_form( $order_id, $redirect = true ) {
		global $woocommerce;
		if ( function_exists( 'wc_get_order' ) ) {
			$order = wc_get_order( $order_id );
		} else {
			$order = new WC_Order( $order_id );
		}

		if ( 'yes' == $this->testmode ) {
			$borgun_adr = self::BORGUN_ENDPOINT_SANDBOX . 'default.aspx';
		} else {
			$borgun_adr = self::BORGUN_ENDPOINT_LIVE . 'default.aspx';
		}
		$borgun_args = $this->get_borgun_args( $order );
		$borgun_args_array = array();
		foreach ( $borgun_args as $key => $value ) {
			$borgun_args_array[] = '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_attr( $value ) . '" />';
		}

		if($redirect){
			$redirecttext = $this->get_translated_string($this->redirecttext, 'redirecttext');
			$code = sprintf( '$.blockUI({
					message: "%s",
					baseZ: 99999,
					overlayCSS: { background: "#fff", opacity: 0.6 },
					css: {
						padding:        "20px",
						zindex:         "9999999",
						textAlign:      "center",
						color:          "#555",
						border:         "3px solid #aaa",
						backgroundColor:"#fff",
						cursor:         "wait",
						lineHeight:     "24px",
					}
				});

				jQuery("#borgun_payment_form").submit();', $redirecttext );
			wc_enqueue_js( $code );
		}

		$cancel_btn_html = ( current_user_can( 'cancel_order', $order_id ) ) ? '<a class="button cancel" href="' . htmlspecialchars_decode($order->get_cancel_order_url()) . '">' . __( 'Cancel order &amp; restore cart', 'borgun_woocommerce' ) . '</a>' : '';
		$html_form = '<form action="' . esc_url( $borgun_adr ) . '" method="post" id="borgun_payment_form">'
					 . implode( '', $borgun_args_array )
					 . '<input type="submit" class="button" id="wc_submit_borgun_payment_form" value="' . __( 'Pay via Teya', 'borgun_woocommerce' ) . '" /> ' . $cancel_btn_html . '</form>';

		return $html_form;
	}

	function process_payment( $order_id ) {
		if ( function_exists( 'wc_get_order' ) ) {
			$order = wc_get_order( $order_id );
		} else {
			$order = new WC_Order( $order_id );
		}

		return array(
			'result'   => 'success',
			'redirect' => $order->get_checkout_payment_url( true )
		);
	}

	/**
	 * Check API payment 'payment' step response
	 *
	 */
	public function handle_webhook_callback(){
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		if( empty($_POST) ){
			status_header(400);
			exit('Invalid webhook data');
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$sanitized_data = $this->recursive_sanitize($_POST);
		/* translators: %s: Teya Posted data */
		self::log( sprintf( __( 'handle_webhook_callback, posted: %s', 'borgun_woocommerce' ), wc_print_r($sanitized_data, true) ) );
		$posted = [
			'status' => isset($sanitized_data['status']) ? $sanitized_data['status'] : '',
			'orderid' => isset($sanitized_data['orderid']) ? $sanitized_data['orderid'] : '',
			'orderhash' => isset($sanitized_data['orderhash']) ? $sanitized_data['orderhash'] : '',
			'authorizationcode' => isset($sanitized_data['authorizationcode']) ? $sanitized_data['authorizationcode'] : '',
			'maskedcardnumber' => isset($sanitized_data['maskedcardnumber']) ? $sanitized_data['maskedcardnumber'] : '',
			'refundid' => isset($sanitized_data['refundid']) ? $sanitized_data['refundid'] : '',
			'step'=> isset($sanitized_data['step']) ? $sanitized_data['step'] : '',
			'reference'=> isset($sanitized_data['reference']) ? $sanitized_data['reference'] : '',
			'errorcode'=> isset($sanitized_data['errorcode']) ? $sanitized_data['errorcode'] : '',
			'errordescription'=> isset($sanitized_data['errordescription']) ? $sanitized_data['errordescription'] : ''
		];

		$order = NULL;
		if ( ! empty( $posted['orderid'] ) ) {
			$order_id = NULL;
			$order_id = apply_filters( 'borgun_'.$this->id.'_get_order_id', $order_id, $posted );
			if ( $order_id === NULL ) {
				$order_id = (int) str_replace( 'WC-', '', $posted['orderid'] );
			}
			if( !empty($order_id) ) {
				if ( function_exists( 'wc_get_order' ) ) {
					$order = wc_get_order( $order_id );
				} else {
					$order = new WC_Order( $order_id );
				}
			}
		}

		if(empty($order)){
			status_header(400);
			exit('Order not found');
		}

		$payment_method = $order->get_payment_method();
		if( $payment_method != $this->id ){
			status_header(400);
			exit('Order payment method doesn\'t match');
		}

		$hash = $this->check_order_hash( $order );
		if ( $hash != $posted['orderhash'] ) {
			status_header(400);
			exit('Order hash doesn\'t match');
		}

		if ( $posted['status'] == 'OK' ) {
			if( !$order->is_paid() ){
				// Complete order by confirmation
				$this->payment_status_completed($order, $posted);

				$xml_data = '<?xml version="1.0" encoding="UTF-8"?>';
				$xml_data .= '<PaymentNotification>Accepted</PaymentNotification>';
				header( "Content-Type: application/xml; charset=utf-8" );
				//phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo $xml_data;
				exit();
			}
		}
	}

	/**
	 * Redirect to configured success page
	 *
	 */
	public function success_page_redirect(){
		if(!empty( $this->successurl ) ){
			wp_safe_redirect( $this->successurl );
			exit;
		}
	}

	/**
	 * Check API payment 'confirmation' response
	 *
	 */
	public function check_confirmation_borgun_response($order) {
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		if( empty($_POST) ){
			return;
		}

		$payment_method = $order->get_payment_method();
		if( $payment_method != $this->id ) return;

		if($order->is_paid()) return;

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$sanitized_data = $this->recursive_sanitize($_POST);
		/* translators: %s: Teya Posted data */
		self::log( sprintf( __( 'thank_you_check_borgun_response, posted: %s', 'borgun_woocommerce' ), wc_print_r($sanitized_data, true) ) );
		$posted = [
			'status' => isset($sanitized_data['status']) ? $sanitized_data['status'] : '',
			'orderid' => isset($sanitized_data['orderid']) ? $sanitized_data['orderid'] : '',
			'orderhash' => isset($sanitized_data['orderhash']) ? $sanitized_data['orderhash'] : '',
			'authorizationcode' => isset($sanitized_data['authorizationcode']) ? $sanitized_data['authorizationcode'] : '',
			'maskedcardnumber' => isset($sanitized_data['maskedcardnumber']) ? $sanitized_data['maskedcardnumber'] : '',
			'refundid' => isset($sanitized_data['refundid']) ? $sanitized_data['refundid'] : '',
			'step'=> isset($sanitized_data['step']) ? $sanitized_data['step'] : '',
			'reference'=> isset($sanitized_data['reference']) ? $sanitized_data['reference'] : '',
			'errorcode'=> isset($sanitized_data['errorcode']) ? $sanitized_data['errorcode'] : '',
			'errordescription'=> isset($sanitized_data['errordescription']) ? $sanitized_data['errordescription'] : ''
		];


		$posted_order_id = ( !empty($posted['orderid']) ) ? (int) str_replace( 'WC-', '', $posted['orderid'] ) : '';
		if ( $posted_order_id != $order->get_id() ) return;

		$hash = $this->check_order_hash( $order );
		if ( $hash != $posted['orderhash'] ) {
			$order->add_order_note( __( 'Order hash doesn\'t match', 'borgun_woocommerce' ) );
			return;
		}

		if ( $posted['status'] == 'OK') {
			// Complete order
			$this->payment_status_completed($order, $posted);
		}
	}

	/**
	 * Handle a completed payment.
	 *
	 * @param WC_Order $order  Order object.
	 * @param array    $posted Posted data.
	 */
	protected function payment_status_completed( $order, $posted ) {
		global $woocommerce;
		/* translators: %s: Teya Posted data */
		self::log( sprintf( __( 'payment_status_completed: %s', 'borgun_woocommerce' ), wc_print_r($posted, true) ) );
		$order_metas = array(
			'authorizationcode'=> $posted['authorizationcode'],
			'maskedcardnumber'=> $posted['maskedcardnumber'],
			'refundid'=> $posted['refundid']
		);
		$this->save_order_metas( $order, $order_metas );
		$message = __( 'Teya payment completed', 'borgun_woocommerce' );
		$order->add_order_note($message);
		$order->payment_complete($posted['authorizationcode']);
		$woocommerce->cart->empty_cart();
	}

	function receipt_page( $order_id ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if(isset($_REQUEST['status'])){
			$posted = array();
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$posted['status'] = sanitize_text_field( wp_unslash($_REQUEST['status']) );
			if($posted['status'] == 'ERROR'){
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo $this->generate_borgun_form( $order_id, false);
			}
		}else{
			$receipttext = $this->get_translated_string($this->receipttext, 'receipttext');
			if( !empty($receipttext) ) printf('<p>%s</p>', esc_html($receipttext) );
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo $this->generate_borgun_form( $order_id);
		}
	}

	/**
	 * Round prices.
	 * @param  double $price
	 * @param  WC_Order $order
	 * @return double
	 */
	protected function round( $price, $order ) {
		$precision = 2;

		if ( ! $this->currency_has_decimals( $order->get_currency() ) ) {
			$precision = 0;
		}

		return round( $price, $precision );
	}

	/**
	 * Check if currency has decimals.
	 * @param  string $currency
	 * @return bool
	 */
	protected function currency_has_decimals( $currency ) {
		if ( in_array( $currency, array( 'HUF', 'JPY', 'TWD', 'ISK' ) ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Check tax display.
	 * @return bool
	 */
	protected function tax_display() {
		$prices_include_tax = wc_tax_enabled() ? get_option( 'woocommerce_prices_include_tax' ) : 'yes';
		return ( $prices_include_tax === 'yes' ) ? true : false ;
	}

	/**
	 * Check fee tax display.
	 * @param  WC_Order_Item_Fee $item
	 * @return bool
	 */
	protected function fee_tax_display( $item ) {
		$tax_display = $item->get_tax_status();
		return ( $tax_display == 'taxable' ) ? true : false ;
	}

	/**
	 * Adds text keys to_translate.
	 * @param  $text_keys array
	 * @return array
	 */
	public function borgun_text_keys_to_translate( $text_keys ){
		if( !in_array( 'receipttext', $text_keys ) )
			$text_keys[] = 'receipttext';
		if( !in_array( 'redirecttext', $text_keys ) )
			$text_keys[] = 'redirecttext';

		return $text_keys;
	}

	/**
	 * Getting translated value
	 * @param  $string string Original field value
	 * @param  $name string Field key
	 * @return string
	 */
	public function get_translated_string( $string, $name ) {
		$translated_string = $string;
		// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
		$current_lang = apply_filters( 'wpml_current_language', NULL );
		if($current_lang && class_exists('WCML_WC_Gateways') ){
			if(defined('WCML_WC_Gateways::STRINGS_CONTEXT')){
				$domain = WCML_WC_Gateways::STRINGS_CONTEXT;
			} else {
				$domain = 'woocommerce';
			}
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			$translated_string = apply_filters( 'wpml_translate_single_string', $string, $domain, $this->id . '_gateway_' . $name, $current_lang );
		}

		return $translated_string;
	}

	/**
	 * Get cancel order url
	 */
	function checkout_payment_handler(){
		global $wp;
		$order_id = '';
		if( !empty( $wp->query_vars['order-pay']) ){
			$order_id = (int) $wp->query_vars['order-pay'];
		}

		if(empty($order_id)) return;

		if ( function_exists( 'wc_get_order' ) ) {
			$order = wc_get_order( $order_id );
		} else {
			$order = new WC_Order( $order_id );
		}

		if(empty($order)) return;

		if($order->get_payment_method() != $this->id) return;

		if($order->is_paid()) return;

		$borgun_settings = get_option('woocommerce_borgun_settings');

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if (isset($_REQUEST['status'])) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if($_REQUEST['status'] === 'ERROR'){
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$errorcode = ( isset($_REQUEST['errorcode']) && !empty( $_REQUEST['errorcode'] ) ) ? sanitize_text_field( wp_unslash($_REQUEST['errorcode']) ) : '';
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$error = ( isset($_REQUEST['errordescription']) && !empty( $_REQUEST['errordescription'] ) ) ? sanitize_text_field( wp_unslash($_REQUEST['errordescription']) ) : __('Payment error','borgun_woocommerce');
				if(!empty($errorcode))
					$error .= '(errorcode:' . $errorcode . ')';

				$order->add_order_note( $error );
				wc_add_notice( $error, 'error' );
				if( !empty($borgun_settings) && !empty( $borgun_settings['errorurl'] ) ){
					$redirect = $borgun_settings['errorurl'];
					$redirect = $borgun_settings['errorurl'];
					wp_safe_redirect($redirect);
					exit;
				}
			}
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			elseif($_REQUEST['status'] === 'CANCEL'){
				if(current_user_can( 'cancel_order', $order->get_id() )){
					$message = __('Payment canceled by the customer','borgun_woocommerce');
					$order->add_order_note($message);
					$redirect = htmlspecialchars_decode($order->get_cancel_order_url());
					wp_safe_redirect( $redirect );
					exit;
				}
			}
		}
	}

	/**
	 * Save order metas
	 * @since 1.0.0
	 * @param WC_Order $order The order which is in a transitional state.
	 * @param array $meta Response meta data
	 */
	public function save_order_metas($order, $metas ){
		if( !empty($metas) ){
			foreach ($metas as $key => $meta) {
				if( !empty($meta) ) $order->update_meta_data( '_' . $this->id . '_' . $key, $meta );
			}
			$order->save();

		}
	}

	/**
	 * Process refund.
	 *
	 * @param  int        $order_id Order ID.
	 * @param  float|null $amount Refund amount.
	 * @param  string     $reason Refund reason.
	 * @return boolean True or false based on success, or a WP_Error object.
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {
		if ( function_exists( 'wc_get_order' ) ) {
			$order = wc_get_order( $order_id );
		} else {
			$order = new WC_Order( $order_id );
		}
		if ( ! $this->can_refund_order( $order ) ) {
			return new WP_Error( 'error', __( 'Refund failed.', 'borgun_woocommerce' ) );
		}

		$refundid = $order->get_meta('_' . $this->id . '_refundid');

		if ( 'yes' == $this->testmode ) {
			$borgun_adr = self::BORGUN_ENDPOINT_SANDBOX . 'refund.aspx';
		} else {
			$borgun_adr = self::BORGUN_ENDPOINT_LIVE . 'refund.aspx';
		}

		$data = [];
		$data['RefundId'] = $refundid;
		$data['MerchantId'] = $this->merchantid;
		$data['PaymentGatewayId'] = $this->paymentgatewayid;
		$data['Checkhash'] = $this->check_order_refund_hash( $order );
		$data['amount'] = $amount;

		$response = wp_remote_post(
			$borgun_adr,
			array(
				'method'      => 'POST',
				'timeout'     => 45,
				'redirection' => 5,
				'httpversion' => '1.0',
				'headers'     => array(),
				'body'        => $data,
				'sslverify'   => false
			)
		);

		if ( empty( $response['body'] ) ) {
			$message = __( 'Empty Response', 'borgun_woocommerce');
			$order->add_order_note( $message );
			return new WP_Error( 'error', $message );
		} elseif ( is_wp_error( $response ) ) {
			$message = $response->get_error_message();
			$order->add_order_note( $message );
			return new WP_Error( 'error', $message );
		}

		$body = wp_remote_retrieve_body( $response );
		if( !empty($body) ) {
			parse_str($body, $result);
			if( isset($result['action_code']) && $result['action_code'] == '000' && isset($result['ret']) && $result['ret'] == true ) {
				/* translators: 1: Refund amount 2: Order currency */
				$message = sprintf( __('Refunded %1$s %2$s via Teya', 'borgun_woocommerce' ), $amount, $order->get_currency() );
				$order->add_order_note( $message );
				return true;
			}
			else {
				/* translators: 1: Refund reason 2: Refund amount 3: Order currency */
				$message = sprintf( __('Teya error: %1$s, Amount: %2$s %3$s', 'borgun_woocommerce' ), $result['message'], $amount, $order->get_currency() );
				$order->add_order_note( $message );
				return new WP_Error( 'error', $result['message'] );
			}
		}

		return false;
	}


	/**
	 * Get price_decimals depending on multi-currency settings
	 *
	 * @param  string $currency
	 * 
	 * @return int
	*/
	private function get_price_decimals($currency) {
		global $woocommerce_wpml;

		$decimals = wc_get_price_decimals();

		// Override price decimals depending on woocommerce-multilingual multi-currency settings
		if( !empty($woocommerce_wpml) &&
			defined('WCML_MULTI_CURRENCIES_INDEPENDENT') &&
			WCML_MULTI_CURRENCIES_INDEPENDENT === (int) $woocommerce_wpml->settings['enable_multi_currency']
		) {
			$currency_options = $woocommerce_wpml->get_setting( 'currency_options' );
			if( !empty($currency_options) &&
				isset($currency_options[$currency]) &&
				isset($currency_options[$currency]['num_decimals'])
			){
				$decimals = $currency_options[$currency]['num_decimals'];
			}
		}

		return apply_filters('borgun_price_decimals', $decimals, $currency);
	}

	/**
	 * Sanitize values
	 */
	private function recursive_sanitize( $data ) {
		if ( is_array( $data ) || is_object( $data ) ) {
			foreach ( $data as $key => &$value ) {
				$key = sanitize_key( $key );
				$value = $this->recursive_sanitize( $value );
			}
			return $data;
		} else {
			if(is_string($data)){
				return sanitize_text_field( $data );
			}else{
				return (float)$data;
			}
		}
	}
}